<?php
// Optional: PHP-Wrapper, damit du später leicht Includes einbauen kannst
?>
<!doctype html>
<html lang="de">
<head>
<meta charset="utf-8">
<title>Multi-Timer &amp; Countdown Dashboard – TheCodeAtlas Demo</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link rel="stylesheet"
      href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css">
<style>
.timer-card {
    border-left: 4px solid #e5e7eb;
    transition: background-color .2s, border-color .2s;
}
.timer-running {
    border-color: #22c55e;
    background: #ecfdf3;
}
.timer-paused {
    border-color: #eab308;
    background: #fffbeb;
}
.timer-finished {
    border-color: #ef4444;
    background: #fef2f2;
}
.timer-time {
    font-size: 1.8rem;
    font-weight: 700;
    font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
}
.badge-status {
    font-size: .75rem;
    text-transform: uppercase;
    letter-spacing: .08em;
}
</style>
</head>
<body class="bg-light">

<div class="container py-5">
    <h1 class="fw-bold mb-1">Multi-Timer &amp; Countdown Dashboard</h1>
    <p class="text-muted mb-4">
        Lege mehrere Countdown-Timer an, starte/pause sie einzeln oder alle gemeinsam – ideal für Streams,
        Veranstaltungen, Pausen-Slots oder Fokus-Sessions.
    </p>

    <!-- Optional: Breadcrumbs -->
    <?php /*
    include $_SERVER['DOCUMENT_ROOT'] . "/includes/breadcrumbs.php";
    */ ?>

    <!-- Steuerung / Timer anlegen -->
    <div class="card shadow-sm mb-4">
        <div class="card-body">
            <h5 class="mb-3">Neuen Timer hinzufügen</h5>
            <form id="addTimerForm" class="row gy-2 gx-2 align-items-end">
                <div class="col-md-5">
                    <label class="form-label">Name des Timers</label>
                    <input type="text"
                           id="timerName"
                           class="form-control"
                           placeholder="z. B. Pause, Talk 1, Q&amp;A"
                           required>
                </div>
                <div class="col-4 col-md-2">
                    <label class="form-label">Minuten</label>
                    <input type="number"
                           id="timerMinutes"
                           class="form-control"
                           min="0" max="999"
                           value="5">
                </div>
                <div class="col-4 col-md-2">
                    <label class="form-label">Sekunden</label>
                    <input type="number"
                           id="timerSeconds"
                           class="form-control"
                           min="0" max="59"
                           value="0">
                </div>
                <div class="col-4 col-md-3 d-grid">
                    <button class="btn btn-primary">
                        Timer hinzufügen
                    </button>
                </div>
            </form>

            <hr class="my-4">

            <div class="d-flex flex-wrap gap-2">
                <button id="btnStartAll" class="btn btn-success btn-sm">
                    ▶ Alle starten
                </button>
                <button id="btnPauseAll" class="btn btn-warning btn-sm">
                    ⏸ Alle pausieren
                </button>
                <button id="btnResetAll" class="btn btn-outline-secondary btn-sm">
                    ⏮ Alle zurücksetzen
                </button>
            </div>

            <p class="small text-muted mt-3 mb-0">
                Hinweis: Die Timer laufen im Browser. Wenn du die Seite schließt oder neu lädst,
                werden die Timer zurückgesetzt.
            </p>
        </div>
    </div>

    <!-- Timer-Liste -->
    <div id="timerList" class="row g-3">
        <!-- Timerkarten werden per JS eingefügt -->
    </div>

</div>

<script>
(function() {
    let timers = [];
    let nextId = 1;
    let tickInterval = null;

    const timerListEl  = document.getElementById('timerList');
    const addForm      = document.getElementById('addTimerForm');
    const inputName    = document.getElementById('timerName');
    const inputMinutes = document.getElementById('timerMinutes');
    const inputSeconds = document.getElementById('timerSeconds');

    const btnStartAll  = document.getElementById('btnStartAll');
    const btnPauseAll  = document.getElementById('btnPauseAll');
    const btnResetAll  = document.getElementById('btnResetAll');

    function formatTime(totalSec) {
        if (totalSec < 0) totalSec = 0;
        const h = Math.floor(totalSec / 3600);
        const m = Math.floor((totalSec % 3600) / 60);
        const s = totalSec % 60;

        const hh = h > 0 ? String(h).padStart(2, '0') + ':' : '';
        const mm = String(m).padStart(2, '0') + ':';
        const ss = String(s).padStart(2, '0');
        return hh + mm + ss;
    }

    function renderTimers() {
        timerListEl.innerHTML = '';

        if (timers.length === 0) {
            timerListEl.innerHTML = `
                <div class="col-12">
                    <div class="text-center text-muted py-4">
                        Noch keine Timer angelegt.
                    </div>
                </div>
            `;
            return;
        }

        timers.forEach(timer => {
            const col = document.createElement('div');
            col.className = 'col-md-6 col-lg-4';

            const card = document.createElement('div');
            card.className = 'card timer-card';
            card.dataset.id = timer.id;

            if (timer.state === 'running') {
                card.classList.add('timer-running');
            } else if (timer.state === 'paused') {
                card.classList.add('timer-paused');
            } else if (timer.state === 'finished') {
                card.classList.add('timer-finished');
            }

            const body = document.createElement('div');
            body.className = 'card-body';

            const headerRow = document.createElement('div');
            headerRow.className = 'd-flex justify-content-between align-items-start mb-2';

            const title = document.createElement('div');
            title.innerHTML = `<strong>${escapeHtml(timer.name)}</strong>`;

            const statusBadge = document.createElement('span');
            statusBadge.className = 'badge badge-status';

            if (timer.state === 'running') {
                statusBadge.classList.add('bg-success', 'text-white');
                statusBadge.textContent = 'Läuft';
            } else if (timer.state === 'paused') {
                statusBadge.classList.add('bg-warning', 'text-dark');
                statusBadge.textContent = 'Pausiert';
            } else if (timer.state === 'finished') {
                statusBadge.classList.add('bg-danger', 'text-white');
                statusBadge.textContent = 'Fertig';
            } else {
                statusBadge.classList.add('bg-secondary', 'text-white');
                statusBadge.textContent = 'Bereit';
            }

            headerRow.appendChild(title);
            headerRow.appendChild(statusBadge);

            const timeEl = document.createElement('div');
            timeEl.className = 'timer-time mb-2';
            timeEl.textContent = formatTime(timer.remainingSec);

            const footerRow = document.createElement('div');
            footerRow.className = 'd-flex justify-content-between align-items-center';

            const btnGroup = document.createElement('div');
            btnGroup.className = 'btn-group btn-group-sm';

            const btnStartPause = document.createElement('button');
            btnStartPause.type = 'button';
            btnStartPause.className = 'btn btn-outline-primary';
            btnStartPause.textContent = timer.state === 'running' ? 'Pause' : 'Start';
            btnStartPause.addEventListener('click', () => toggleTimer(timer.id));

            const btnReset = document.createElement('button');
            btnReset.type = 'button';
            btnReset.className = 'btn btn-outline-secondary';
            btnReset.textContent = 'Reset';
            btnReset.addEventListener('click', () => resetTimer(timer.id));

            const btnDelete = document.createElement('button');
            btnDelete.type = 'button';
            btnDelete.className = 'btn btn-outline-danger';
            btnDelete.textContent = 'Löschen';
            btnDelete.addEventListener('click', () => deleteTimer(timer.id));

            btnGroup.appendChild(btnStartPause);
            btnGroup.appendChild(btnReset);
            btnGroup.appendChild(btnDelete);

            footerRow.appendChild(btnGroup);

            body.appendChild(headerRow);
            body.appendChild(timeEl);
            body.appendChild(footerRow);

            card.appendChild(body);
            col.appendChild(card);
            timerListEl.appendChild(col);
        });
    }

    function escapeHtml(str) {
        return str.replace(/[&<>"']/g, function(m) {
            switch (m) {
                case '&': return '&amp;';
                case '<': return '&lt;';
                case '>': return '&gt;';
                case '"': return '&quot;';
                case "'": return '&#039;';
            }
            return m;
        });
    }

    function addTimer(name, totalSec) {
        if (totalSec <= 0) {
            alert('Bitte eine Dauer größer 0 Sekunden wählen.');
            return;
        }

        timers.push({
            id: nextId++,
            name: name,
            durationSec: totalSec,
            remainingSec: totalSec,
            state: 'ready'
        });
        renderTimers();
        ensureTick();
    }

    function toggleTimer(id) {
        const timer = timers.find(t => t.id === id);
        if (!timer) return;

        if (timer.state === 'running') {
            timer.state = 'paused';
        } else {
            if (timer.remainingSec <= 0) {
                timer.remainingSec = timer.durationSec;
            }
            timer.state = 'running';
        }
        renderTimers();
        ensureTick();
    }

    function resetTimer(id) {
        const timer = timers.find(t => t.id === id);
        if (!timer) return;
        timer.remainingSec = timer.durationSec;
        timer.state = 'ready';
        renderTimers();
    }

    function deleteTimer(id) {
        timers = timers.filter(t => t.id !== id);
        renderTimers();
        if (timers.length === 0) {
            stopTick();
        }
    }

    function startAll() {
        timers.forEach(t => {
            if (t.remainingSec <= 0) {
                t.remainingSec = t.durationSec;
            }
            t.state = 'running';
        });
        renderTimers();
        ensureTick();
    }

    function pauseAll() {
        timers.forEach(t => {
            if (t.state === 'running') {
                t.state = 'paused';
            }
        });
        renderTimers();
    }

    function resetAll() {
        timers.forEach(t => {
            t.remainingSec = t.durationSec;
            t.state = 'ready';
        });
        renderTimers();
    }

    function ensureTick() {
        if (tickInterval !== null) return;
        tickInterval = setInterval(tick, 1000);
    }

    function stopTick() {
        if (tickInterval !== null) {
            clearInterval(tickInterval);
            tickInterval = null;
        }
    }

    function tick() {
        let anyRunning = false;

        timers.forEach(t => {
            if (t.state === 'running') {
                anyRunning = true;
                t.remainingSec--;
                if (t.remainingSec <= 0) {
                    t.remainingSec = 0;
                    t.state = 'finished';
                }
            }
        });

        if (!anyRunning) {
            stopTick();
        }
        renderTimers();
    }

    // Events
    addForm.addEventListener('submit', function(e) {
        e.preventDefault();
        const name = inputName.value.trim() || 'Timer ' + nextId;
        let minutes = parseInt(inputMinutes.value, 10);
        let seconds = parseInt(inputSeconds.value, 10);

        if (isNaN(minutes) || minutes < 0) minutes = 0;
        if (isNaN(seconds) || seconds < 0) seconds = 0;
        if (seconds > 59) seconds = 59;

        const totalSec = minutes * 60 + seconds;
        addTimer(name, totalSec);

        inputName.value = '';
    });

    btnStartAll.addEventListener('click', startAll);
    btnPauseAll.addEventListener('click', pauseAll);
    btnResetAll.addEventListener('click', resetAll);

    // Initial-Render
    renderTimers();
})();
</script>
<footer class="text-center mt-5 py-4 border-top small text-muted">
    © <?= date('Y') ?> <a href="https://thecodeatlas.net" target="_blank" rel="noopener">
        TheCodeAtlas.net
    </a> – Kostenloses Multi-Timer Tool.
    Weitergabe und Einbau in eigene Projekte erlaubt, solange der Hinweis auf TheCodeAtlas erhalten bleibt.
</footer>

</body>
</html>
